/*
 * Copyright 2016-2022 the original author or authors.
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v2.0 which
 * accompanies this distribution and is available at
 *
 * http://www.eclipse.org/legal/epl-v20.html
 */

package org.junitpioneer.jupiter;

import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static org.junit.jupiter.api.Assertions.fail;
import static org.junitpioneer.testkit.assertion.PioneerAssert.assertThat;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import org.assertj.core.api.Condition;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junitpioneer.testkit.ExecutionResults;
import org.junitpioneer.testkit.PioneerTestKit;
import org.opentest4j.AssertionFailedError;
import org.opentest4j.TestAbortedException;

@DisplayName("ExpectedToFail extension")
public class ExpectedToFailExtensionTests {

	@Test
	void abortsOnTestFailure() {
		ExecutionResults results = PioneerTestKit.executeTestMethod(ExpectedToFailTestCases.class, "failure");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				.hasMessage("Test marked as temporarily 'expected to fail' failed as expected")
				.hasCause(new AssertionFailedError("failed"));
	}

	@Test
	void abortsOnTestFailureWithMetaAnnotation() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailTestCases.class, "metaAnnotationFailure");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				.hasMessage("Test marked as temporarily 'expected to fail' failed as expected")
				.hasCause(new AssertionFailedError("failed"));
	}

	@Test
	void abortsOnTestFailureWithMessage() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailTestCases.class, "failureWithMessage");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				.hasMessage("Custom message")
				.hasCause(new AssertionFailedError("failed"));
	}

	@Test
	void abortsOnException() {
		ExecutionResults results = PioneerTestKit.executeTestMethod(ExpectedToFailTestCases.class, "exception");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				.hasMessage("Test marked as temporarily 'expected to fail' failed as expected")
				.hasCause(new RuntimeException("test"));
	}

	@Test
	void preservesTestAbort() {
		ExecutionResults results = PioneerTestKit.executeTestMethod(ExpectedToFailTestCases.class, "aborted");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				// Ignore message prefix generated by JUnit
				.hasMessageEndingWith("custom assumption message");
	}

	@Test
	void failsOnWorkingTest() {
		ExecutionResults results = PioneerTestKit.executeTestMethod(ExpectedToFailTestCases.class, "working");
		assertThat(results)
				.hasSingleStartedTest()
				.whichFailed()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("Test marked as 'expected to fail' succeeded; remove @ExpectedToFail from it");
	}

	@Test
	void doesNotAbortOnBeforeEachTestFailure() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailFailureBeforeEachTestCases.class, "test");
		assertThat(results)
				.hasSingleFailedTest()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("before each");
	}

	@Test
	void doesNotAbortOnAfterEachTestFailure() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailFailureAfterEachTestCases.class, "test");
		assertThat(results)
				.hasSingleFailedTest()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("Test marked as 'expected to fail' succeeded; remove @ExpectedToFail from it")
				// Note: This check for suppressed exception actually tests JUnit platform behavior
				.hasSuppressedException(new AssertionFailedError("after each"));
	}

	@Test
	void doesNotAbortOnAfterEachTestFailureAfterTestAbort() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailAbortThenFailureAfterEachTestCases.class, "test");
		assertThat(results)
				.hasSingleFailedTest()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("after each");
	}

	@Test
	void afterEachAbortAfterTestFailure() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailFailureThenAbortAfterEachTestCases.class, "test");
		assertThat(results)
				.hasSingleStartedTest()
				.whichAborted()
				.withExceptionInstanceOf(TestAbortedException.class)
				.hasMessage("Test marked as temporarily 'expected to fail' failed as expected")
				.hasCause(new AssertionFailedError("failed"))
				// Note: This check for suppressed exception actually tests JUnit platform behavior
				.has(new Condition<>((Throwable throwable) -> {
					Throwable[] suppressed = throwable.getSuppressed();
					return suppressed.length == 1 && suppressed[0] instanceof TestAbortedException
					// Ignore message prefix generated by JUnit
							&& suppressed[0].getMessage().endsWith("custom assumption message");
				}, "suppressed JUnit abort exception"));
	}

	@Test
	void doesNotAbortOnBeforeAllTestFailure() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailFailureBeforeAllTestCases.class, "test");
		assertThat(results).hasNumberOfStartedTests(0);
		assertThat(results)
				.hasSingleFailedContainer()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("before all");
	}

	@Test
	void doesNotAbortOnAfterAllTestFailure() {
		ExecutionResults results = PioneerTestKit
				.executeTestMethod(ExpectedToFailFailureAfterAllTestCases.class, "test");
		assertThat(results)
				.hasSingleStartedTest()
				.whichFailed()
				.withExceptionInstanceOf(AssertionError.class)
				.hasMessage("Test marked as 'expected to fail' succeeded; remove @ExpectedToFail from it");
	}

	static class ExpectedToFailTestCases {

		@ExpectedToFail
		@Retention(RUNTIME)
		@Target(METHOD)
		@interface ExpectedToFailMetaAnnotation {
		}

		@Test
		@ExpectedToFail
		void failure() {
			fail("failed");
		}

		@Test
		@ExpectedToFailMetaAnnotation
		void metaAnnotationFailure() {
			fail("failed");
		}

		@Test
		@ExpectedToFail("Custom message")
		void failureWithMessage() {
			fail("failed");
		}

		@Test
		@ExpectedToFail
		void exception() {
			throw new RuntimeException("test");
		}

		@Test
		@ExpectedToFail
		void aborted() {
			// Assumption should have higher precedence than @ExpectedToFail
			Assumptions.assumeTrue(false, "custom assumption message");
		}

		@Test
		@ExpectedToFail
		void working() {
			// Does not cause failure or error
		}

	}

	/**
	 * {@link BeforeEach} should not be considered by {@link ExpectedToFail} because it
	 * is not specific to the annotated test method.
	 */
	static class ExpectedToFailFailureBeforeEachTestCases {

		@BeforeEach
		void beforeEach() {
			fail("before each");
		}

		@Test
		@ExpectedToFail
		void test() {
		}

	}

	/**
	 * {@link AfterEach} should be considered by {@link ExpectedToFail} because it
	 * might fail due to changes made to the test instance by the test method.
	 */
	static class ExpectedToFailFailureAfterEachTestCases {

		@AfterEach
		void afterEach() {
			fail("after each");
		}

		@Test
		@ExpectedToFail
		void test() {
		}

	}

	/**
	 * Abort in test method followed by failure in {@link AfterEach} method should
	 * be treated as expected failure.
	 */
	static class ExpectedToFailAbortThenFailureAfterEachTestCases {

		@AfterEach
		void afterEach() {
			fail("after each");
		}

		@Test
		@ExpectedToFail
		void test() {
			Assumptions.assumeTrue(false, "custom assumption message");
		}

	}

	static class ExpectedToFailFailureThenAbortAfterEachTestCases {

		@AfterEach
		void afterEach() {
			Assumptions.assumeTrue(false, "custom assumption message");
		}

		@Test
		@ExpectedToFail
		void test() {
			fail("failed");
		}

	}

	/**
	 * {@link BeforeAll} should not be considered by {@link ExpectedToFail}.
	 */
	static class ExpectedToFailFailureBeforeAllTestCases {

		@BeforeAll
		static void beforeAll() {
			fail("before all");
		}

		@Test
		@ExpectedToFail
		void test() {
		}

	}

	/**
	 * {@link AfterAll} should not be considered by {@link ExpectedToFail}.
	 */
	static class ExpectedToFailFailureAfterAllTestCases {

		@AfterAll
		static void afterAll() {
			fail("after all");
		}

		@Test
		@ExpectedToFail
		void test() {
		}

	}

}
