// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Cross section for e+e- > gamma eta
  class SND_2023_I2670980 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(SND_2023_I2670980);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      book(_numEtaGamma, 1, 1, 1);
      for (const string& en : _numEtaGamma.binning().edges<0>()) {
        const size_t idx = en.find(" to ");
        if (idx != string::npos) {
          const double emin = stod(en.substr(0,idx));
          const double emax = stod(en.substr(idx+4,string::npos));
          if (inRange(sqrtS()/GeV, emin, emax)) {
            _sqs = en; break;
          }
        }
        else {
          const double eval = stod(en)*GeV;
          if (isCompatibleWithSqrtS(eval)) {
            _sqs = en; break;
          }
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }

      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::pid==221)) {
        if (p.children().empty()) continue;
        // find the eta
        map<long,int> nRes = nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        // eta gamma
        if (ncount!=1) continue;
        bool matched = true;
        for (const auto& val : nRes) {
          if (val.first==22) {
            if (val.second !=1) {
              matched = false;
              break;
            }
          }
          else if (val.second!=0) {
            matched = false;
            break;
          }
        }
        if (matched) {
          _numEtaGamma->fill(_sqs);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_numEtaGamma, crossSection()/sumOfWeights()/picobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _numEtaGamma;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(SND_2023_I2670980);

}
