#!/usr/bin/env python

"""
Free/busy-related scheduling functionality.

Copyright (C) 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.data import uri_values
from imiptools.dates import ValidityError, to_timezone
from imiptools.handlers.scheduling.common import standard_responses

def schedule_in_freebusy(handler, args, freebusy=None):

    """
    Attempt to schedule the current object of the given 'handler' in the
    free/busy schedule of a resource, returning an indication of the kind of
    response to be returned.

    The 'args' are used to configure the behaviour of the function. If a number
    is indicated, it is taken to mean the number of concurrent events that can
    be scheduled at the same point in time, with the default being 1.

    If 'freebusy' is specified, the given collection of busy periods will be
    used to determine whether any conflicts occur. Otherwise, the current user's
    free/busy records will be used.
    """

    # Permit the indication that concurrent events may be scheduled.

    try:
        max_concurrent = int((args[1:] or ["1"])[0])
    except ValueError:
        max_concurrent = 1

    # If newer than any old version, discard old details from the
    # free/busy record and check for suitability.

    periods = handler.get_periods(handler.obj)

    freebusy = freebusy or handler.get_store().get_freebusy(handler.user)
    offers = handler.get_store().get_freebusy_offers(handler.user)

    # Check the periods against any scheduled events and against
    # any outstanding offers.

    if max_concurrent == 1:
        scheduled = handler.can_schedule(freebusy, periods)
        scheduled = scheduled and handler.can_schedule(offers, periods)
    else:
        conflicts = get_scheduling_conflicts(handler, freebusy, [handler.user])
        scheduled = conflicts[handler.user] < max_concurrent
        conflicts = get_scheduling_conflicts(handler, offers, [handler.user])
        scheduled = scheduled and conflicts[handler.user] < max_concurrent

    return standard_responses(handler, scheduled and "ACCEPTED" or "DECLINED")

def schedule_corrected_in_freebusy(handler, args):

    """
    Attempt to schedule the current object of the given 'handler', correcting
    specified datetimes according to the configuration of a resource,
    returning an indication of the kind of response to be returned.

    The 'args' are used to configure the behaviour of the function. If a number
    is indicated, it is taken to mean the number of concurrent events that can
    be scheduled at the same point in time, with the default being 1.
    """

    obj = handler.obj.copy()

    # Check any constraints on the request.

    try:
        corrected = handler.correct_object()

    # Refuse to schedule obviously invalid requests.

    except ValidityError:
        return None

    # With a valid request, determine whether the event can be scheduled.

    scheduled = schedule_in_freebusy(handler, args)
    response, description = scheduled or ("DECLINED", None)

    # Restore the original object if it was corrected but could not be
    # scheduled.

    if response == "DECLINED" and corrected:
        handler.set_object(obj)
    
    # Where the corrected object can be scheduled, issue a counter
    # request.

    response = response == "ACCEPTED" and (corrected and "COUNTER" or "ACCEPTED") or "DECLINED"
    return standard_responses(handler, response)

def schedule_next_available_in_freebusy(handler, args):

    """
    Attempt to schedule the current object of the given 'handler', correcting
    specified datetimes according to the configuration of a resource, then
    suggesting the next available period in the free/busy records if scheduling
    cannot occur for the requested period, returning an indication of the kind
    of response to be returned.

    The 'args' are used to configure the behaviour of the function. If a number
    is indicated, it is taken to mean the number of concurrent events that can
    be scheduled at the same point in time, with the default being 1.
    """

    _ = handler.get_translator()

    scheduled = schedule_corrected_in_freebusy(handler, args)
    response, description = scheduled or ("DECLINED", None)

    if response in ("ACCEPTED", "COUNTER"):
        return scheduled

    # There should already be free/busy information for the user.

    user_freebusy = handler.get_store().get_freebusy(handler.user)

    # Maintain a separate copy of the data.

    busy = user_freebusy.copy()

    # Subtract any periods from this event from the free/busy collections.

    event_periods = handler.remove_from_freebusy(busy)

    # Find busy periods for the other attendees.

    for attendee in uri_values(handler.obj.get_values("ATTENDEE")):
        if attendee != handler.user:

            # Get a copy of the attendee's free/busy data.

            freebusy = handler.get_store().get_freebusy_for_other(handler.user, attendee).copy()
            if freebusy:
                freebusy.remove_periods(event_periods)
                busy += freebusy

    # Obtain the combined busy periods.

    busy = busy.coalesce_freebusy()

    # Obtain free periods.

    free = busy.invert_freebusy()
    permitted_values = handler.get_permitted_values()
    periods = []

    # Do not attempt to redefine rule-based periods.

    last = None

    for period in handler.get_periods(handler.obj, explicit_only=True):
        duration = period.get_duration()

        # Try and schedule periods normally since some of them may be
        # compatible with the schedule.

        if permitted_values:
            period = period.get_corrected(permitted_values)

        scheduled = handler.can_schedule(busy, [period])

        if scheduled:
            periods.append(period)
            last = period.get_end()
            continue

        # Get free periods from the time of each period.

        for found in free.periods_from(period):

            # Skip any periods before the last period.

            if last:
                if last > found.get_end():
                    continue

                # Adjust the start of the free period to exclude the last period.

                found = found.make_corrected(max(found.get_start(), last), found.get_end())

            # Only test free periods long enough to hold the requested period.

            if found.get_duration() >= duration:

                # Obtain a possible period, starting at the found point and
                # with the requested duration. Then, correct the period if
                # necessary.

                start = to_timezone(found.get_start(), period.get_tzid())
                possible = period.make_corrected(start, start + period.get_duration())
                if permitted_values:
                    possible = possible.get_corrected(permitted_values)

                # Only if the possible period is still within the free period
                # can it be used.

                if possible.within(found):
                    periods.append(possible)
                    break

        # Where no period can be found, decline the invitation.

        else:
            return "DECLINED", _("The recipient is unavailable in the requested period.")

        # Use the found period to set the start of the next window to search.

        last = periods[-1].get_end()

    # Replace the periods in the object.

    obj = handler.obj.copy()
    changed = handler.obj.set_periods(periods)

    # Check one last time, reverting the change if not scheduled.

    scheduled = schedule_in_freebusy(handler, args, busy)
    response, description = scheduled or ("DECLINED", None)

    if response == "DECLINED":
        handler.set_object(obj)

    response = response == "ACCEPTED" and (changed and "COUNTER" or "ACCEPTED") or "DECLINED"
    return standard_responses(handler, response)

# Registry of scheduling functions.

scheduling_functions = {
    "schedule_in_freebusy" : [schedule_in_freebusy],
    "schedule_corrected_in_freebusy" : [schedule_corrected_in_freebusy],
    "schedule_next_available_in_freebusy" : [schedule_next_available_in_freebusy],
    }

# Registries of locking and unlocking functions.

locking_functions = {}
unlocking_functions = {}

# Registries of listener functions.

confirmation_functions = {}
retraction_functions = {}

# vim: tabstop=4 expandtab shiftwidth=4
